(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 12.1' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[     24776,        615]
NotebookOptionsPosition[     19932,        522]
NotebookOutlinePosition[     21336,        570]
CellTagsIndexPosition[     21293,        567]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[TextData[{
 "Pick Events and Delaunay Algorithm\n",
 StyleBox["\tby Konrad Polthier\n\t",
  FontSize->16],
 StyleBox["(v1.12, 2020-03-19)",
  FontSize->16,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}]
}], "Chapter",
 CellGroupingRules->"SectionGrouping",
 CellChangeTimes->{3.79376990105614*^9},
 TextAlignment->Left,ExpressionUUID->"491096b8-7786-4873-91d2-fb0a56ed698a"],

Cell[CellGroupData[{

Cell[TextData[{
 "This notebook demonstrates a use of real-time heavy computations by ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " while interacting with the JavaView display. We show a planar \
triangulation and drag vertices around while ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " simultaneously calculates the Delaunay triangulation."
}], "Text",ExpressionUUID->"820ca491-4e1a-4eb6-b51f-3329733a787d"],

Cell[TextData[{
 "When the JavaView display is open, then drag a vertex of the geometry and \
wait a second. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will catch the update event of the geometry and recompute the \
triangulation to remain Delaunay. This notebook does not fulfill 'real-time' \
but I think it can be tuned and will try so."
}], "Text",ExpressionUUID->"bd661def-f76b-41c0-9f63-a5e3c0eb06c0"],

Cell[TextData[{
 "Other Experiments:\nInitially the major mode of the display is \
PvDisplayIf`MODE_PICK which means that vertices may be picked and dragged.\n\
1. After some vertex dragging you might want to try to add another vertex. \
Just keep pressing the a-key and pick somewhere to add another vertex (even \
outside the geometry). Since the geometry is updated after this event the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " handler is also called and recomputes the new triangulation.\n2. Or delete \
some vertices: keep pressing the m-key and drag a rectangle to mark all \
vertices inside. Then open the control window using ctrl-s or via popup menu \
and choose the menu entry Method -> Delete -> Remove Marked Vertices."
}], "Text",ExpressionUUID->"c163336b-c410-4676-9da9-1b3594d64512"],

Cell["\<\
The notebook is a little bit longer since I rewrote one command in the \
package DiscreteMath`ComputationalGeometry` which does not handle vertices \
with triple valence correctly.\
\>", "Text",ExpressionUUID->"f0eb4b32-61f3-479b-bd20-73cb0b6e1bbd"],

Cell["\<\
I use the asynchronous event handling to get a smoother perception when \
dragging the mouse.\
\>", "Text",ExpressionUUID->"7fbc9615-0ab2-420e-b71a-92f4ea8d7d06"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "Load Java runtime into ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " and load JavaView."
}], "MathCaption",ExpressionUUID->"843df3b2-b0ab-4cb0-83af-5fb175de27f1"],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"<<", "JavaView`JLink`"}], "\n", 
 RowBox[{
  RowBox[{"viewer", "=", 
   RowBox[{"InstallJavaView", "[", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",ExpressionUUID->"5936cf6b-22e3-4c85-b038-26421d6fbb7e"],

Cell[BoxData[
 TemplateBox[{
  "General", "obs", 
   "\"\\!\\(\\*RowBox[{\\\"\\\\\\\"WorldPlot`\\\\\\\"\\\"}]\\) has been \
superseded by \\!\\(\\*RowBox[{\\\"GeoGraphics\\\"}]\\), and is now obsolete. \
It will not be included in future versions of the Wolfram Language.\"", 2, 1, 
   1, 33085239449095655655, "Local"},
  "MessageTemplate"]], "Message", "MSG",
 CellChangeTimes->{3.793781853661066*^9},
 CellLabel->
  "During evaluation of \
In[1]:=",ExpressionUUID->"f8b2318a-0c03-4b08-b8b2-52846b0a441f"],

Cell[BoxData["\<\"JavaView`RunThrough` v3.0.3\"\>"], "Print",
 CellChangeTimes->{3.7937818539902782`*^9},
 CellLabel->
  "During evaluation of \
In[1]:=",ExpressionUUID->"138cafb6-db53-4a2e-b097-ca5621dbbd9e"],

Cell[BoxData["\<\"JavaView`JLink` v3.0.3\"\>"], "Print",
 CellChangeTimes->{3.793781854034986*^9},
 CellLabel->
  "During evaluation of \
In[1]:=",ExpressionUUID->"1e3afd04-1030-4c39-a646-a81c93611bb9"]
}, Open  ]],

Cell["Load Computational Geometry Package.", "MathCaption",ExpressionUUID->"ba6ab18d-1486-4b4e-9d2c-ac2a8c483faa"],

Cell[BoxData[
 RowBox[{"<<", "ComputationalGeometry`"}]], "Input",
 CellChangeTimes->{{3.7937822312252965`*^9, 3.7937822363225636`*^9}},
 CellLabel->"In[5]:=",ExpressionUUID->"98a28863-54da-4051-b2aa-b22c0465eb62"],

Cell["\<\
Get the current JavaView display and select pick mode in xy-projection.\
\>", "MathCaption",ExpressionUUID->"a08ab387-4ccc-4d66-95e0-2bcb388330ac"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"disp", "=", 
   RowBox[{"viewer", "@", 
    RowBox[{"getDisplay", "[", "]"}]}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"disp", "@", 
   RowBox[{"selectCamera", "[", "PvDisplayIf`CAMERAUORTHOUXY", "]"}]}], ";", 
  RowBox[{"disp", "@", 
   RowBox[{"setMajorMode", "[", "PvDisplayIf`MODEUPICK", "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[6]:=",ExpressionUUID->"ce788ef5-279e-4388-8e4f-cf87da81fea7"],

Cell["\<\
Create an empty geometry which will become the planar point set.\
\>", "MathCaption",ExpressionUUID->"fb53fe49-0de0-43e4-bbd5-8dc9f4cbe7b8"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"geom", " ", "=", " ", 
    RowBox[{"JavaNew", "[", 
     RowBox[{"\"\<jv.geom.PgElementSet\>\"", ",", "3"}], "]"}]}], ";"}], 
  " "}]], "Input",
 CellLabel->"In[8]:=",ExpressionUUID->"12f05c53-ee87-4826-b91d-49a1b98d93f4"],

Cell["\<\
Compute a sample plane mesh and register it inside the display.\
\>", "MathCaption",ExpressionUUID->"39bbdac5-9a14-4203-8d86-9b8955b08998"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"geom", "@", 
   RowBox[{"computePlane", "[", 
    RowBox[{"4", ",", "4", ",", "0.", ",", "0.", ",", "2.", ",", "1."}], 
    "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"geom", "@", 
   RowBox[{"setName", "[", "\"\<Delaunay Triangulation\>\"", "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"geom", "@", 
   RowBox[{"showIndices", "[", "True", "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"geom", "@", 
   RowBox[{"showVertices", "[", "True", "]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"JavaView", "[", "geom", "]"}], ";"}]}], "Input",
 CellLabel->"In[9]:=",ExpressionUUID->"bcfd1fc2-d859-4ffe-bf65-c22a000562ea"],

Cell[CellGroupData[{

Cell["\<\
The next function will be called upon pick and drag events. We do not use the \
mouse event since we just operate on the geometry one of whose vertex has \
been changed by the vertex dragging. Given a xy-planar geometry this method \
computes a Delaunay triangulation and updates the geometry. Method makes use \
of the Computational Geometry package of Mathematica.\
\>", "Text",ExpressionUUID->"0e278bb6-8b78-44a6-ac81-9f40afd22e83"],

Cell["\<\
Compute a Delaunay triangulation of a given point set upon pick events.\
\>", "MathCaption",ExpressionUUID->"84b279b3-b082-412c-bc50-12075c75a02a"],

Cell[BoxData[
 RowBox[{
  RowBox[{"delaunay", "[", "geom_", "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
      RowBox[{"PsDebug`message", "[", 
       RowBox[{"\"\<Start Delaunay ...\>\"", "+", 
        RowBox[{
         RowBox[{"JavaNew", "[", "\"\<java.util.Date\>\"", "]"}], "@", 
         RowBox[{"toString", "[", "]"}]}]}], "]"}], ";"}], "*)"}], 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"vertex", " ", "=", " ", 
      RowBox[{"geom", "@", 
       RowBox[{"getVertices", "[", "]"}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"data3d", "=", " ", 
      RowBox[{"PdVector`getEntries", "[", "vertex", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"dim", " ", "=", 
      RowBox[{"geom", "@", 
       RowBox[{"getDimOfVertices", "[", "]"}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"vertexData2d", " ", "=", " ", 
      RowBox[{"Partition", "[", 
       RowBox[{
        RowBox[{"Flatten", "[", "data3d", "]"}], ",", "2", ",", "dim"}], 
       "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"(*", " ", 
      RowBox[{
       RowBox[{
       "This", " ", "gives", " ", "the", " ", "counterclockwise", " ", 
        "vertex", " ", "adjacency", " ", "list", " ", "for", " ", "each", " ",
         "point", " ", "in", " ", "the", " ", "Delaunay", " ", 
        RowBox[{"triangulation", ".", " ", "For"}], " ", "example", " ", 
        "the", " ", "entry", 
        RowBox[{
         StyleBox["{", "MR"], 
         RowBox[{
          StyleBox["1", "MR"], 
          StyleBox[",", "MR"], " ", 
          RowBox[{
           StyleBox["{", "MR"], 
           RowBox[{
            StyleBox["4", "MR"], 
            StyleBox[",", "MR"], " ", 
            StyleBox["3", "MR"], 
            StyleBox[",", "MR"], " ", 
            StyleBox["2", "MR"]}], 
           StyleBox["}", "MR"]}]}], 
         StyleBox["}", "MR"]}], "indicates", " ", "that", " ", "the", " ", 
        "first", " ", "point", " ", "in", 
        StyleBox["data2D", "MR"], "is", " ", "connected", " ", "in", " ", 
        "counterclockwise", " ", "order", " ", "to", " ", "the", " ", 
        "fourth"}], ",", "third", ",", 
       RowBox[{"and", " ", "second", " ", 
        RowBox[{"points", "."}]}]}], " ", "*)"}], "\[IndentingNewLine]", 
     RowBox[{"val", "=", 
      RowBox[{"DelaunayTriangulation", "[", "vertexData2d", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"elemData", "=", 
      RowBox[{"adjacenyListToTriangulation", "[", "val", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"element", "=", 
      RowBox[{"PiVector`copyNew", "[", "elemData", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"geom", "@", 
      RowBox[{"setNumElements", "[", 
       RowBox[{"Length", "[", "element", "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"geom", "@", 
      RowBox[{"setElements", "[", "element", "]"}]}], ";", 
     "\[IndentingNewLine]", "elemData"}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[14]:=",ExpressionUUID->"097f60f1-3d30-43e6-8bb6-1c2031de1176"],

Cell["\<\
Helper module to convert a vertex adjacency list to connectivity of a \
triangulation.\
\>", "MathCaption",ExpressionUUID->"9a796025-3fae-404c-a357-035bcf4dda53"],

Cell[BoxData[
 RowBox[{
  RowBox[{"adjacenyListToTriangulation", "[", 
   RowBox[{"val", ":", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"_Integer", ",", 
        RowBox[{"{", 
         RowBox[{"_Integer", ".."}], "}"}]}], "}"}], ".."}], "}"}]}], "]"}], ":=",
   "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "nov", ",", "i", ",", "j", ",", "k", ",", "adj", ",", "vali", ",", "a1", 
      ",", "a2", ",", "a1Adj", ",", "vala1", ",", "elem"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"nov", " ", "=", " ", 
      RowBox[{"Length", "[", "val", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"elem", " ", "=", " ", 
      RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"For", "[", 
      RowBox[{
       RowBox[{"i", "=", "1"}], ",", 
       RowBox[{"i", "<=", "nov"}], ",", 
       RowBox[{"i", "++"}], ",", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"adj", " ", "=", " ", 
         RowBox[{
          RowBox[{"val", "[", 
           RowBox[{"[", "i", "]"}], "]"}], "[", 
          RowBox[{"[", "2", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"vali", " ", "=", " ", 
         RowBox[{"Length", "[", "adj", "]"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"For", "[", 
         RowBox[{
          RowBox[{"j", "=", "1"}], ",", 
          RowBox[{"j", "<=", "vali"}], ",", 
          RowBox[{"j", "++"}], ",", "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"a1", " ", "=", " ", 
            RowBox[{
             RowBox[{"adj", "[", 
              RowBox[{"[", "j", "]"}], "]"}], "+", "0"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"a2", " ", "=", 
            RowBox[{
             RowBox[{"adj", "[", 
              RowBox[{"[", 
               RowBox[{
                RowBox[{"Mod", "[", 
                 RowBox[{"j", ",", "vali"}], "]"}], "+", "1"}], "]"}], "]"}], 
             "+", "0"}]}], ";", "\[IndentingNewLine]", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"a1", "<", "i"}], " ", "||", 
              RowBox[{"a2", "<", "i"}]}], ",", 
             RowBox[{"Continue", "[", "]"}]}], "]"}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"a1Adj", "=", 
            RowBox[{
             RowBox[{"val", "[", 
              RowBox[{"[", "a1", "]"}], "]"}], "[", 
             RowBox[{"[", "2", "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
           RowBox[{"vala1", "=", 
            RowBox[{"Length", "[", "a1Adj", "]"}]}], ";", 
           "\[IndentingNewLine]", 
           RowBox[{"For", "[", 
            RowBox[{
             RowBox[{"k", "=", "1"}], ",", 
             RowBox[{"k", "<=", "vala1"}], ",", 
             RowBox[{"k", "++"}], ",", "\[IndentingNewLine]", 
             RowBox[{
              RowBox[{"If", "[", 
               RowBox[{
                RowBox[{
                 RowBox[{"a1Adj", "[", 
                  RowBox[{"[", "k", "]"}], "]"}], "!=", "a2"}], ",", 
                RowBox[{"Continue", "[", "]"}]}], "]"}], ";", 
              "\[IndentingNewLine]", 
              RowBox[{"elem", "=", 
               RowBox[{"Append", "[", 
                RowBox[{"elem", ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"i", "-", "1"}], ",", 
                   RowBox[{"a1", "-", "1"}], ",", 
                   RowBox[{"a2", "-", "1"}]}], "}"}]}], "]"}]}]}]}], 
            "\[IndentingNewLine]", "]"}], ";"}]}], "\[IndentingNewLine]", 
         "]"}], ";"}]}], "\[IndentingNewLine]", "]"}], ";", 
     "\[IndentingNewLine]", "elem"}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[15]:=",ExpressionUUID->"35ee28d6-676b-4dea-b03e-b8fd8d5e23e0"],

Cell["\<\
This function will be called upon update events of the geometry. We do not \
use the mouse event since we just operate on the geometry one of whose vertex \
has been changed by the vertex dragging. Print timing messages into the \
JavaView console.\
\>", "MathCaption",ExpressionUUID->"4e6ae2da-3e7e-43a1-abb4-4ccfcdb46a85"],

Cell[BoxData[
 RowBox[{
  RowBox[{"updateDelaunay", "[", 
   RowBox[{"evt_", ",", "message_"}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"JavaBlock", "[", "\[IndentingNewLine]", 
   RowBox[{"(*", 
    RowBox[{
     RowBox[{"PsDebug`message", "[", 
      RowBox[{
       RowBox[{"JavaNew", "[", 
        RowBox[{"\"\<java.lang.String\>\"", ",", "\"\<Start: \>\""}], "]"}], 
       "@", 
       RowBox[{"concat", "[", "date", "]"}]}], "]"}], ";"}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"delaunay", "[", "geom", "]"}], ";", "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
      RowBox[{"PsDebug`message", "[", 
       RowBox[{
        RowBox[{"JavaNew", "[", 
         RowBox[{"\"\<java.lang.String\>\"", ",", "\"\<End:   \>\""}], "]"}], 
        "@", 
        RowBox[{"concat", "[", "date", "]"}]}], "]"}], ";"}], " ", "*)"}], 
    "\[IndentingNewLine]", 
    RowBox[{"(*", " ", 
     RowBox[{
      RowBox[{"Redraw", " ", "geometry", " ", "in", " ", 
       RowBox[{"display", ".", " ", "Do"}], " ", "not", " ", "call", " ", 
       RowBox[{"geom", "@", 
        RowBox[{"update", "[", "geom", "]"}]}], " ", "to", " ", "avoid", " ", 
       "getting", " ", "into", " ", "a", " ", "loop"}], ",", " ", 
      RowBox[{
      "since", " ", "this", " ", "handler", " ", "is", " ", "invoked", " ", 
       "on", " ", "every", " ", "update", " ", "of", " ", 
       RowBox[{"geom", "."}]}]}], " ", "*)"}], "\[IndentingNewLine]", 
    RowBox[{"disp", "@", 
     RowBox[{"update", "[", "geom", "]"}]}], ";"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{
     RowBox[{"Do", " ", "not", " ", "release", " ", "event"}], ",", " ", 
     RowBox[{"as", " ", "one", " ", "is", " ", "usually", " ", "advised"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{"since", " ", "this", " ", "event", " ", "is", " ", "the", " ", 
      RowBox[{"geometry", " ", "'"}], 
      RowBox[{
       RowBox[{"geom", "'"}], ".", " ", "If"}], " ", "I", " ", "release", " ",
       "evt", " ", "here", " ", 
      RowBox[{"then", " ", "'"}], 
      RowBox[{"geom", "'"}], " ", "becomes", " ", "a", " ", "removed", " ", 
      "object", " ", "and", " ", "further", " ", "access", " ", "to", " ", 
      "it", " ", "leads", " ", "to", " ", "error", " ", 
      RowBox[{"messages", "."}]}]}], " ", "*)"}], "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"ReleaseObject", "[", "evt", "]"}], " ", "*)"}], 
   "\[IndentingNewLine]", "]"}]}]], "Input",
 CellLabel->"In[16]:=",ExpressionUUID->"77f61007-5a86-4cfb-ad36-6db4fac254a1"]
}, Open  ]],

Cell["\<\
Register an update listener of the geometry. We do not use the mouse events \
in the display since we just operate on the geometry one of whose vertex has \
been changed by the vertex dragging.\
\>", "MathCaption",ExpressionUUID->"1f909bda-7364-43c9-b159-b1e405af3a28"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"listener", " ", "=", " ", 
   RowBox[{"JavaNew", "[", "\"\<jvx.math.MathUpdateListener\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"listener", "@", 
   RowBox[{"setHandler", "[", 
    RowBox[{"\"\<update\>\"", ",", " ", "\"\<updateDelaunay\>\""}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"geom", "@", 
   RowBox[{"addUpdateListener", "[", "listener", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[17]:=",ExpressionUUID->"db0a4dbe-2718-4bc1-95df-2fb603624aa9"],

Cell["\<\
Let us print the current time to the JavaView console (press ctrl-d inside \
the JavaView display to open the console) to measure the time spent inside \
the Delaunay computation.\
\>", "MathCaption",ExpressionUUID->"4088ddc1-db8c-4831-9a1b-e3abd50ed9df"],

Cell[BoxData[
 RowBox[{
  RowBox[{"date", " ", ":=", 
   RowBox[{
    RowBox[{"JavaNew", "[", "\"\<java.util.Date\>\"", "]"}], "@", 
    RowBox[{"toString", "[", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[20]:=",ExpressionUUID->"fdea40be-c84a-4b6f-bb97-ea501aeecf4e"],

Cell[TextData[{
 "After calling ShareKernel[] start dragging vertices around in the display. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will compute the Delaunay triangulation in a different thread in the \
background."
}], "MathCaption",ExpressionUUID->"9ee57f03-8669-47d8-ac71-5d164970e56b"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ShareKernel", "[", "]"}]], "Input",
 CellLabel->
  "In[188]:=",ExpressionUUID->"5677c0e7-20a5-4d05-9af2-e55d127cbaf5"],

Cell[BoxData["8"], "Output",
 CellLabel->
  "Out[188]=",ExpressionUUID->"f0314d0c-f3b0-4508-9863-10fec48fd36d"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "When done with interaction, then call UnshareKernel[]. After calling \
UnshareKernel[] ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " will no longer receive events from ",
 StyleBox["JavaView",
  FontSlant->"Italic"],
 " unless ShareKernel[] is called again."
}], "MathCaption",ExpressionUUID->"28028bea-559a-48b7-a44d-47aff983be13"],

Cell[BoxData[
 RowBox[{"UnshareKernel", "[", "]"}]], "Input",
 CellLabel->
  "(sharing) \
In[189]:=",ExpressionUUID->"c88b1a6f-cd1e-4efd-aed5-295735ce2326"]
}, Open  ]]
},
ScreenStyleEnvironment->"Brackets",
CellGrouping->Manual,
WindowSize->{782, 962},
WindowMargins->{{Automatic, 17}, {Automatic, 0}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PageHeaders->{{
   Cell[
    TextData[{
      CounterBox["Page"]}], "PageNumber"], Inherited, 
   Cell[
    TextData[{"Konrad Polthier"}], "PageHeader"]}, {
   Cell[
    TextData[{"The ", 
      StyleBox["JavaView", FontSlant -> "Italic"], " Packages"}], 
    "PageHeader"], Inherited, 
   Cell[
    TextData[{
      CounterBox["Page"]}], "PageNumber"]}},
PageFooters->{{Inherited, 
   Cell[
    TextData[{
     "\[Copyright]1999 Wolfram Research, Inc. All rights reserved."}], 
    "PageFooter"], Inherited}, {Inherited, 
   Cell[
    TextData[{
     "\[Copyright]1999 Wolfram Research, Inc. All rights reserved."}], 
    "PageFooter"], Inherited}},
PageFooterLines->{True, True},
PrintingOptions->{"FacingPages"->True,
"PrintCellBrackets"->True,
"PrintMultipleHorizontalPages"->False,
"PrintRegistrationMarks"->False,
"PrintingMargins"->{{56.6875, 56.6875}, {56.6875, 56.6875}}},
FrontEndVersion->"12.1 for Microsoft Windows (64-bit) (March 14, 2020)",
StyleDefinitions->"HelpBrowser.nb",
ExpressionUUID->"3f2a5f4b-1db5-4e8d-88c9-ed64556b34f2"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[558, 20, 399, 11, 94, "Chapter",ExpressionUUID->"491096b8-7786-4873-91d2-fb0a56ed698a",
 CellGroupingRules->"SectionGrouping"],
Cell[CellGroupData[{
Cell[982, 35, 426, 9, 26, "Text",ExpressionUUID->"820ca491-4e1a-4eb6-b51f-3329733a787d"],
Cell[1411, 46, 419, 8, 28, "Text",ExpressionUUID->"bd661def-f76b-41c0-9f63-a5e3c0eb06c0"],
Cell[1833, 56, 812, 12, 99, "Text",ExpressionUUID->"c163336b-c410-4676-9da9-1b3594d64512"],
Cell[2648, 70, 259, 4, 18, "Text",ExpressionUUID->"f0eb4b32-61f3-479b-bd20-73cb0b6e1bbd"],
Cell[2910, 76, 172, 3, 18, "Text",ExpressionUUID->"7fbc9615-0ab2-420e-b71a-92f4ea8d7d06"]
}, Open  ]],
Cell[CellGroupData[{
Cell[3119, 84, 189, 5, 30, "MathCaption",ExpressionUUID->"843df3b2-b0ab-4cb0-83af-5fb175de27f1"],
Cell[CellGroupData[{
Cell[3333, 93, 233, 5, 32, "Input",ExpressionUUID->"5936cf6b-22e3-4c85-b038-26421d6fbb7e"],
Cell[3569, 100, 507, 11, 17, "Message",ExpressionUUID->"f8b2318a-0c03-4b08-b8b2-52846b0a441f"],
Cell[4079, 113, 209, 4, 16, "Print",ExpressionUUID->"138cafb6-db53-4a2e-b097-ca5621dbbd9e"],
Cell[4291, 119, 202, 4, 16, "Print",ExpressionUUID->"1e3afd04-1030-4c39-a646-a81c93611bb9"]
}, Open  ]],
Cell[4508, 126, 114, 0, 30, "MathCaption",ExpressionUUID->"ba6ab18d-1486-4b4e-9d2c-ac2a8c483faa"],
Cell[4625, 128, 214, 3, 20, "Input",ExpressionUUID->"98a28863-54da-4051-b2aa-b22c0465eb62"],
Cell[4842, 133, 157, 2, 30, "MathCaption",ExpressionUUID->"a08ab387-4ccc-4d66-95e0-2bcb388330ac"],
Cell[5002, 137, 449, 11, 44, "Input",ExpressionUUID->"ce788ef5-279e-4388-8e4f-cf87da81fea7"],
Cell[5454, 150, 150, 2, 30, "MathCaption",ExpressionUUID->"fb53fe49-0de0-43e4-bbd5-8dc9f4cbe7b8"],
Cell[5607, 154, 269, 7, 20, "Input",ExpressionUUID->"12f05c53-ee87-4826-b91d-49a1b98d93f4"],
Cell[5879, 163, 149, 2, 30, "MathCaption",ExpressionUUID->"39bbdac5-9a14-4203-8d86-9b8955b08998"],
Cell[6031, 167, 712, 18, 70, "Input",ExpressionUUID->"bcfd1fc2-d859-4ffe-bf65-c22a000562ea"],
Cell[CellGroupData[{
Cell[6768, 189, 445, 6, 28, "Text",ExpressionUUID->"0e278bb6-8b78-44a6-ac81-9f40afd22e83"],
Cell[7216, 197, 157, 2, 30, "MathCaption",ExpressionUUID->"84b279b3-b082-412c-bc50-12075c75a02a"],
Cell[7376, 201, 3160, 75, 222, "Input",ExpressionUUID->"097f60f1-3d30-43e6-8bb6-1c2031de1176"],
Cell[10539, 278, 172, 3, 30, "MathCaption",ExpressionUUID->"9a796025-3fae-404c-a357-035bcf4dda53"],
Cell[10714, 283, 3805, 95, 272, "Input",ExpressionUUID->"35ee28d6-676b-4dea-b03e-b8fd8d5e23e0"],
Cell[14522, 380, 334, 5, 41, "MathCaption",ExpressionUUID->"4e6ae2da-3e7e-43a1-abb4-4ccfcdb46a85"],
Cell[14859, 387, 2559, 56, 171, "Input",ExpressionUUID->"77f61007-5a86-4cfb-ad36-6db4fac254a1"]
}, Open  ]],
Cell[17433, 446, 279, 4, 30, "MathCaption",ExpressionUUID->"1f909bda-7364-43c9-b159-b1e405af3a28"],
Cell[17715, 452, 506, 13, 44, "Input",ExpressionUUID->"db0a4dbe-2718-4bc1-95df-2fb603624aa9"],
Cell[18224, 467, 265, 4, 30, "MathCaption",ExpressionUUID->"4088ddc1-db8c-4831-9a1b-e3abd50ed9df"],
Cell[18492, 473, 268, 6, 20, "Input",ExpressionUUID->"fdea40be-c84a-4b6f-bb97-ea501aeecf4e"],
Cell[18763, 481, 305, 6, 30, "MathCaption",ExpressionUUID->"9ee57f03-8669-47d8-ac71-5d164970e56b"],
Cell[CellGroupData[{
Cell[19093, 491, 142, 3, 20, "Input",ExpressionUUID->"5677c0e7-20a5-4d05-9af2-e55d127cbaf5"],
Cell[19238, 496, 111, 2, 19, "Output",ExpressionUUID->"f0314d0c-f3b0-4508-9863-10fec48fd36d"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[19398, 504, 359, 9, 30, "MathCaption",ExpressionUUID->"28028bea-559a-48b7-a44d-47aff983be13"],
Cell[19760, 515, 156, 4, 20, "Input",ExpressionUUID->"c88b1a6f-cd1e-4efd-aed5-295735ce2326"]
}, Open  ]]
}
]
*)

